const { app, BrowserWindow, MenuItem, Menu, session } = require('electron')
const path = require('node:path')
const express = require('express')
const net = require('net') // 用于检查端口是否被占用

let httpServer;
let mainWindow;

// 1. 检查端口是否被占用，避免服务启动失败
function checkPort(port, callback) {
  const server = net.createServer()
    .once('error', (err) => {
      // 端口被占用时尝试下一个端口
      if (err.code === 'EADDRINUSE') {
        callback(port + 1); // 端口+1重试
      } else {
        callback(null); // 其他错误
      }
    })
    .once('listening', () => {
      // 端口可用，关闭临时服务并返回可用端口
      server.close(() => callback(port));
    })
    .listen(port);
}

// 2. 启动HTTP服务（确保服务就绪后再通知）
function startHttpServer(callback) {
  // 先检查3000端口是否可用，不可用则自动递增
  checkPort(3000, (availablePort) => {
    if (!availablePort) {
      console.error('无法启动HTTP服务：端口占用且无法自动分配');
      return callback(null);
    }

    const appExpress = express();
    // 托管dist文件夹（确保路径正确，dist与main.js同级）
    appExpress.use(express.static(path.join(__dirname, 'dist')));
    
    // 启动服务
    httpServer = appExpress.listen(availablePort, () => {
      const serverUrl = `http://localhost:${availablePort}`;
      console.log(`HTTP服务启动成功：${serverUrl}`);
      callback(serverUrl); // 服务就绪，返回访问地址
    });
  });
}

// 3. 创建窗口（仅在服务启动成功后调用）
function createWindow(serverUrl) {
  mainWindow = new BrowserWindow({
    width: 1980,
    height: 1080,
    webPreferences: {
      preload: path.join(__dirname, 'preload.js'),
      contextIsolation: true,
      partition: 'persist:my-app-session'
    }
  });

  // 获取系统默认菜单
  const defaultMenu = Menu.getApplicationMenu();
  let backMenuItem;
  
  if (defaultMenu) {
    // 创建返回菜单项
    backMenuItem = new MenuItem({
      label: '← 返回',
      click: () => {
        // 使用新的API检查是否可以返回并执行返回操作
        if (mainWindow.webContents.navigationHistory.canGoBack()) {
          mainWindow.webContents.navigationHistory.goBack();
        }
      },
      enabled: false // 初始禁用
    });

    // 复制默认菜单结构并插入导航菜单
    const newMenuItems = [...defaultMenu.items];
    newMenuItems.push(backMenuItem);
    
    // 构建新菜单并应用
    const newMenu = Menu.buildFromTemplate(newMenuItems);
    Menu.setApplicationMenu(newMenu);
  }

  // 更新按钮状态的函数 - 使用新的API检查是否可以返回
  function updateBackButton() {
    if (backMenuItem) { // 检查变量是否存在
      backMenuItem.enabled = mainWindow.webContents.navigationHistory.canGoBack();
    }
  }
  
  // 监听页面导航，更新按钮状态
  mainWindow.webContents.on('did-navigate', updateBackButton);
  mainWindow.webContents.on('did-navigate-in-page', updateBackButton);

  // 加载HTTP服务地址
  mainWindow.loadURL(serverUrl);

  // 窗口关闭时停止服务
  mainWindow.on('closed', async () => {
    if (httpServer) httpServer.close();

    mainWindow = null;
  });

  mainWindow.on('close', async (e) => {
  // 阻止窗口立即关闭
  e.preventDefault();
  
  // 先停止HTTP服务
  if (httpServer) httpServer.close();
  
  try {
    // 获取会话引用（此时窗口仍存在）
    const ses = mainWindow.webContents.session;
    
    // 执行清理操作
    await ses.clearStorageData({
      storages: [
        'appcache', 'cookies', 'localstorage', 'sessionstorage',
        'indexdb', 'websql', 'serviceworkers', 'cachestorage'
      ],
      quotas: ['temporary', 'persistent', 'syncable']
    });
    
    await ses.clearCache();
    await ses.clearAuthCache();
  } catch (error) {
  } finally {
    // 清理完成后，手动销毁窗口
    mainWindow.destroy();
    mainWindow = null;
  }
});
}

// 4. 程序入口：先启动服务，成功后再创建窗口
app.whenReady().then(() => {
  // 启动HTTP服务，成功后才创建窗口
  startHttpServer((serverUrl) => {
    if (serverUrl) {
      createWindow(serverUrl); // 服务可用，创建窗口
    } else {
      console.error('服务启动失败，无法创建窗口');
      app.quit(); // 服务启动失败，退出应用
    }

    app.on('activate', () => {
      if (BrowserWindow.getAllWindows().length === 0 && serverUrl) {
        createWindow(serverUrl);
      }
    });
  });
});

// 关闭所有窗口时退出应用
app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    if (httpServer) httpServer.close();
    app.quit();
  }
});
    